/*******************************************************************************/
/* Coq Unit Testing project                                                    */
/* Copyright 2015-2016 Catherine Dubois, Richard Genestier and Alain Giorgetti */
/* Samovar - FEMTO-ST institute                                                */
/*******************************************************************************/
/*      This file is distributed under the terms of the                        */
/*       GNU Lesser General Public License Version 2.1                         */
/*******************************************************************************/

/* File : bet_permut.pl

   Contents:
   1. Validation of permut.pl by counting 
   2. Validation of examples/coq/permut.v by bounded-exhaustive testing (BET). */

:- compile('../../../prolog/measure'). /* For validation by counting. */
:- compile('../../../prolog/cut').     /* For validation by BET. */

:- compile(permut). /* Specification of permutations. */

/* 1. Validation by counting. */

:- write('Validation of line by counting:'), nl, iterate(0,6,line).
% 1,1,2,6,24,120,720, factorials.

/* 2. Validations by BET. */

/* Validation of is_permut, with a reflexive proof. 

   Model:
   Goal (is_permut 3 (list2fun (0::2::1::nil))). apply is_permut_dec. 
   simpl. reflexivity. Qed. */
write_coq_is_permut(PredSym,SizeMax) :-
 increasing(Size,1,SizeMax),
 (settings(fd_on) -> SizeT=Size; nat2term(Size,SizeT)),
 Pred=..[PredSym,L,SizeT],  /* Pred is the generator */
 Pred, 
 write('Goal is_permut '),
 write(Size),
 write(' (list2fun ('), write_list(L), write(')).'), nl,
 write('apply is_permut_dec. simpl. reflexivity. Qed.'), nl,
 flush_output,
 fail. % this causes backtracking
write_coq_is_permut(_,_).
    
  
/* Validation that insertion preserves permutations.
   Model: 
   Eval compute in (is_permutb 4 (insert_fun 3 (list2fun (2::2::0::nil)) 3)). */
write_coq_insert_permut(PredSym,SizeMax)  :-
 increasing(Size,0,SizeMax),
 (settings(fd_on) -> SizeT=Size; nat2term(Size,SizeT)),
 Pred=..[PredSym,L,SizeT],  /* Pred is the generator */
 Pred, 
 increasing(I,0,Size),
 write('Eval compute in (is_permutb '),
 SizeP1 is Size + 1,
 write(SizeP1),
 write(' (insert_fun '),
 write(Size),
 write(' (list2fun ('), write_list(L), write(')) '),
 write(I),
 write(')).'),
 nl, flush_output,
 fail. % this causes backtracking
write_coq_insert_permut(_,_).

/* Validation that direct sum preserves permutations. 
   Model: 
   Eval compute in (is_permutb 5 (sum_fun 
    3 (list2fun (2::2::0::nil))
    2 (list2fun (0::1::nil))
   )).
*/
write_coq_sum_permut(PredSym,SizeMax)  :-
 increasing(Size,0,SizeMax),
 increasing(Size1,0,Size),
 Size2 is Size - Size1,
 (settings(fd_on) -> SizeT1=Size1; nat2term(Size1,SizeT1)),
 Pred1=..[PredSym,L1,SizeT1],  /* Generator of the first list L1 */
 (settings(fd_on) -> SizeT2=Size2; nat2term(Size2,SizeT2)),
 Pred2=..[PredSym,L2,SizeT2],  /* Generator of the second list L2 */
 Pred1,
 Pred2, 
 write('Eval compute in (is_permutb '),
 write(Size),
 write(' (sum_fun '),
 write(Size1),
 write(' (list2fun ('), write_list(L1), write(')) '),
 write(Size2),
 write(' (list2fun ('), write_list(L2), write(')) '),
 write(')).'),
 nl, flush_output,
 fail. % this causes backtracking
write_coq_sum_permut(_,_).



/* One predicate for all the validations of Coq concerning permutations. */
write_coq_permut(PredSym,SizeMax) :-
 write('(* Test suite 1: Validation of is_permut with a reflexive proof. *)'), nl,
 write_coq_is_permut(PredSym,SizeMax), nl, nl,
 write('(* Test suite 2: Validation that insertion preserves permutations (lemma insert_permut). *)'), nl,
 write_coq_insert_permut(PredSym,SizeMax), nl, nl,
 write('(* Test suite 3: Validation that direct sum preserves permutations (lemma sum_permut). *)'), nl,
 write_coq_sum_permut(PredSym,SizeMax),
 nl.


/* Coq file opening and closing, with header and time measures. */
write_coq_open_close_file(Size,PredSym,File) :-
 tell(File),
  write('(* File generated by code in ../permutation/prolog/bet_permut.pl. *)'), nl,
  write('Require Import Arith Arith.Bool_nat Omega List.'), nl,
  write('Require Import endofun permut cut.'), nl, nl, 
  write('Definition is_permutb n f := andb (is_endob n f) (is_injb n f).'), nl, nl,
  ((system_type(sicstus);system_type(swi)) -> statistics(runtime,[T1,_]) ; true),
  ((system_type(gnu)) -> statistics(user_time,[T1,_]) ; true),
  write_coq_permut(PredSym,Size),
  ((system_type(sicstus);system_type(swi)) -> statistics(runtime,[T2,_]) ; true),
  ((system_type(gnu)) -> statistics(user_time,[T2,_]) ; true),
  Time is T2-T1,
  write('(* Time: '), write(Time), write(' ms'), write(' *)'),
 told.

/* Some output before writing in the Coq file. */
write_coq_file(Size,PredSym,File) :-
 write('-- File permut.pl'), nl,
 write('Size: '), write(Size), nl,
 write('Predicate: '), write(PredSym), nl,
 write('Output file: '), write(File), nl,
 write_coq_open_close_file(Size,PredSym,File).
 

:- write_coq_file(4,line,'../../coq/val_permut.v'), halt.
