(*******************************************************************************)
(* Coq Unit Testing project                                                    *)
(* Copyright 2017-2018 Catherine Dubois and Alain Giorgetti                    *)
(* Samovar - FEMTO-ST institute                                                *)
(*******************************************************************************)

(*******************************************************************************)
(*      This file is distributed under the terms of the                        *)
(*       GNU Lesser General Public License Version 2.1                         *)
(*******************************************************************************)

(** Samples and tests with the ROM term generator defined in [qc_rom.v]. *)

Require Import Arith NPeano String Nat Omega.
Require Import prelude cut endofun permut trans operations qc map rom qc_rom.

From QuickChick Require Import QuickChick Tactics.
Require Import String. Open Scope string.

From mathcomp Require Import ssreflect ssrfun ssrbool ssrnat eqtype.

Require Import List.
Import ListNotations.
Import QcDefaultNotation. Open Scope qc_scope.

(* Uncomment the next line for ?? tests instead of 10000: *)
(* Extract Constant Test.defNumTests => "??". *)

From QuickChick Require Import QuickChick Tactics.

Open Scope string_scope.

Require Import mathcomp.ssreflect.ssreflect.

(** * Sampling *)

(** *** Some ROM terms with size 5 *)

Eval compute in "Some ROM terms with size 5".

Sample (genRom 5).

(** Samples with [Sample (genRom 5).]:

 - $B(U0("V"),B("V",U2(B("V","V")))), B(U1(U1(U0(U0("V")))),"V")$, 
 
 $U2(B(U1(B("V","V")), U0("V"))), U6(U6(U2(U1(U0("V")))))$, 
 
 $U7(B(U2(U0("V")),U0("V"))), B(B(U2(B("V","V")),"V"), U0("V"))$, 
 
 $U2(U2(B("V",U1(U0("V"))))), B(U1(U1(U2(B("V","V")))),"V")$, 
 
 $B(B("V",U0("V")),U1(B("V","V"))), B("V",B(B("V",B("V",U0("V"))),"V"))$, 
 
 $B(B(U0("V"),"V"),U0(B("V","V")))$
*)

(*
(** *** Transformation of ROM terms into RSLs representing maps with 5 edges: *)

Sample (genRomMapRslType 5).

(** Samples with [Sample (genRomMapRslType 5).]:

 $[9, 7, 0, 5, 5, 3, 0, 1, 1, 0], [3, 7, 6, 5, 5, 4, 3, 1, 0, 0], [9, 7, 7, 5, 5, 3, 1, 1, 0, 0]$, 
 
 $[3, 7, 5, 6, 4, 4, 2, 1, 0, 0], [0, 7, 3, 5, 4, 4, 3, 1, 1, 0], [9, 7, 1, 5, 2, 3, 3, 2, 0, 0]$,
 
 $[5, 7, 0, 5, 3, 4, 1, 2, 1, 0], [7, 7, 3, 5, 5, 4, 3, 1, 1, 0], [4, 7, 7, 5, 2, 3, 1, 2, 0, 0]$,
 
 $[9, 7, 3, 5, 3, 4, 3, 1, 0, 0], [5, 7, 1, 5, 3, 4, 3, 2, 1, 0]$ *)


(** *** Transformation of ROM terms into local map rotations with 6 edges: *)

Sample (genRomRotationAsListnat 6).

(** Samples with [Sample (genRomRotationAsListnat 6).]:

 $[6, 0, 2, 4, 11, 1, 5, 10, 8, 7, 9, 3], [8, 2, 4, 10, 11, 3, 5, 0, 7, 1, 9, 6]$,

 $[4, 11, 1, 2, 3, 0, 6, 10, 8, 7, 9, 5], [11, 0, 10, 2, 4, 3, 5, 1, 8, 7, 9, 6]$,

 $[8, 0, 6, 11, 3, 10, 5, 2, 7, 4, 9, 1], [8, 10, 1, 0, 6, 4, 5, 3, 7, 2, 9, 11]$,

 $[8, 11, 2, 1, 4, 3, 5, 0, 7, 6, 10, 9], [2, 0, 4, 1, 3, 11, 8, 6, 7, 10, 9, 5]$,

 $[2, 0, 1, 10, 4, 6, 11, 5, 7, 3, 9, 8], [2, 0, 11, 1, 8, 4, 5, 6, 7, 3, 10, 9]$,

 $[11, 0, 6, 8, 4, 10, 5, 2, 7, 3, 9, 1]$
*)

(** * Random Testing *)

Eval compute in "Random testing for ROM terms".

(** * Soundness lemma checking *)

Definition is_transitiveb n (r : list nat) := is_transitive_funb n (list2fun r).

QuickCheck (sized (fun e => 
 forAll (genRomRotationAsListnat e) (fun r => is_transitiveb (2*e) r))).

(** +++ Passed 10000 tests (0 discards) *)
*)

